
//Chris' WinGlide enhancements
//Copr. 1998, Chris Dohnal (cwdohnal@ucdavis.edu)

#include <windows.h>
#include <glide.h>
#include "inifile.h"
#include "dib.h"
#include "msgbox.h"
#include "gamma.h"
#include "pcopy.h"
#include "pointers.h"
#include "main.h"
#include "gammadlg.h"
#include "sysmenu.h"
#include "subclass.h"

BOOL StretchBltWGInitialize(HWND, DWORD, DWORD);
VOID StretchBltWGShutdown(VOID);
VOID StretchBltWGCopy3DfxFrameBuffer(VOID);

static WNDPROC gpPrevWndProc = NULL;
static LRESULT CALLBACK SubclassWndProc(HWND, UINT, WPARAM, LPARAM);

static BOOL CreateFrameUpdateThread(LPDWORD, LPVOID);
static VOID DestroyFrameUpdateThread(VOID);
static DWORD WINAPI FrameUpdateThreadProc(LPVOID);

static HANDLE hFrameUpdateThread = NULL;
static DWORD dwFrameUpdateTID;
static BOOL gbTellFrameUpdateThreadToExit = FALSE;
static HANDLE hMainThreadEvent = NULL;
static HANDLE hFrameUpdateThreadEvent = NULL;
static HANDLE hFrameUpdateThreadExitedEvent = NULL;


BOOL StretchBltWGInitialize(HWND hWnd, DWORD dwWidth, DWORD dwHeight) {
	BOOL bRet;

	//Create the DIB
	bRet = CreateDIB(hWnd, dwWidth, dwHeight);
	if (bRet == FALSE) {
		ErrorLoadingWinGlideMessageBox(NULL, "Could not create DIB section.");

		goto initialization_failure;
	}

	//Allocate memory for the gamma correction table
	bRet = AllocateGammaCorrectionTable();
	if (bRet == FALSE) {
		ErrorLoadingWinGlideMessageBox(NULL, "Could not create gamma correction table.");

		goto initialization_failure;
	}

	//Calculate the gamma correction table
	CalculateGammaCorrectionTable(gRedGamma, gGreenGamma, gBlueGamma);
	
	//See if multi-threading is enabled
	if (gbMultiThreading == TRUE) {
		//Create the frame update thread
		bRet = CreateFrameUpdateThread(&dwFrameUpdateTID, NULL);
		if (bRet == FALSE) {
			ErrorLoadingWinGlideMessageBox(NULL, "Could not create frame update thread.");

			goto initialization_failure;
		}
	}

	//Subclass the window and take care of other modifications to the window
	bRet = CWGSubclassWindow(hWnd, &gpPrevWndProc, SubclassWndProc, WINGLIDE_MODE_STRETCH_BLT);
	if (bRet == FALSE) {
		ErrorLoadingWinGlideMessageBox(NULL, "Window subclassing error.");

		goto initialization_failure;
	}

	return TRUE;

	//Jump here if initialization failed
initialization_failure:

	//Clean up anything that was allocated
	StretchBltWGShutdown();

	return FALSE;
}

VOID StretchBltWGShutdown(VOID) {
	//Unsubclass the window if it was subclassed and clean up other modifications to the window
	CWGUnsubclassWindowIfSubclassed();

	//Destroy the frame update thread
	DestroyFrameUpdateThread();

	//Delete the DIB
	FreeDIB();

	//Free the gamma correction table
	FreeGammaCorrectionTable();
	
	return;
}

VOID StretchBltWGCopy3DfxFrameBuffer(VOID) {
	FxBool fxbRet;
	GrLfbInfo_t grLFBInfo;
	BYTE *p3DfxLFBStart;
	BYTE *pDIB;
	LONG lDIBPitch;
	DWORD *pGammaTable;
	LONG lAlignedWidth;
	BOOL bRet;
	DWORD dwRet;

	//Check the range of these values and make sure they are valid
	if (gdwScreenWidth > gdwGlideScreenWidth) { gdwScreenWidth = gdwGlideScreenWidth; }
	if (gdwScreenHeight > gdwGlideScreenHeight) { gdwScreenHeight = gdwGlideScreenHeight; }
	if (gdwScreenWidth < 0) { gdwScreenWidth = 0; }
	if (gdwScreenHeight < 0) { gdwScreenHeight = 0; }

	//Get information about the DIB
	pDIB = GetDIBBuffer();
	if (pDIB == NULL) {
		return;
	}
	lDIBPitch = GetDIBPitch();

	//Get the pointer to the gamma correction table
	if (gbEnableGammaCorrection == TRUE) {
		pGammaTable = GetGammaCorrectionTablePointer();
		if (pGammaTable == NULL) {
			return;
		}
	}

	//Handle updating the gamma correction table if necessary
	bRet = CheckForNewGammaCorrectionValues(&gRedGamma, &gGreenGamma, &gBlueGamma);
	if (bRet == TRUE) {
		//Calculate the new gamma correction table
		CalculateGammaCorrectionTable(gRedGamma, gGreenGamma, gBlueGamma);
	}

	//See if multi-threading is enabled
	if (gbMultiThreading == TRUE) {
		//Do not continue if the frame update thread does not exist
		if (hFrameUpdateThread == NULL) {
			return;
		}
				
		//Wait for the frame update thread to be ready
		dwRet = WaitForSingleObject(hMainThreadEvent, INFINITE);
	}

	//Lock the 3Dfx LFB
	ZeroMemory(&grLFBInfo, sizeof(grLFBInfo));
	grLFBInfo.size = sizeof(grLFBInfo);
	fxbRet = _grLfbLock(GR_LFB_READ_ONLY, GR_BUFFER_BACKBUFFER, 0,
		GR_ORIGIN_ANY, FXFALSE, &grLFBInfo);
	if (fxbRet == FXFALSE) {
		//Could not lock the 3Dfx LFB
		return;
	}

	//Setup the pointer to the 3Dfx LFB
	if (gOriginLocation == GR_ORIGIN_LOWER_LEFT) {
		p3DfxLFBStart = (BYTE *)grLFBInfo.lfbPtr +
			(grLFBInfo.strideInBytes * (gdwScreenHeight - 1));
	}
	else {
		p3DfxLFBStart = (BYTE *)grLFBInfo.lfbPtr +
			(grLFBInfo.strideInBytes * (gdwGlideScreenHeight - 1));
	}

	//Get the aligned width to copy
	lAlignedWidth = ((gdwScreenWidth + 3) & ~3) >> 2;

	//See if gamma correction is enabled
	if (gbEnableGammaCorrection == TRUE) {
		//Copy the frame from the 3Dfx to the DIB
		if (gbUseMMX == TRUE) {
			MMXGammaPitchCopy8(p3DfxLFBStart, pDIB,
				-(LONG)grLFBInfo.strideInBytes, lDIBPitch,
				lAlignedWidth, gdwScreenHeight,
				pGammaTable);
		}
		else {
			GammaPitchCopy8(p3DfxLFBStart, pDIB,
				-(LONG)grLFBInfo.strideInBytes, lDIBPitch,
				lAlignedWidth, gdwScreenHeight,
				pGammaTable);
		}
	}
	else {
		//Copy the frame from the 3Dfx to the DIB
		if (gbUseMMX == TRUE) {
			MMXPitchCopy8(p3DfxLFBStart, pDIB,
				-(LONG)grLFBInfo.strideInBytes, lDIBPitch,
				lAlignedWidth, gdwScreenHeight);
		}
		else {
			FPUPitchCopy8(p3DfxLFBStart, pDIB,
				-(LONG)grLFBInfo.strideInBytes, lDIBPitch,
				lAlignedWidth, gdwScreenHeight);
		}
	}

	//Unlock the 3Dfx LFB
	fxbRet = _grLfbUnlock(GR_LFB_READ_ONLY, GR_BUFFER_BACKBUFFER);
	if (fxbRet == FXFALSE) {
		//Could not unlock the 3Dfx LFB
		return;
	}

	//Put the frame on the screen
	if (gbMultiThreading == TRUE) {
		//Reset the main thread event
		bRet = ResetEvent(hMainThreadEvent);

		//Resume the frame update thread
		bRet = SetEvent(hFrameUpdateThreadEvent);
	}
	else {
		//Put the DIB on the screen
		DrawStretchedDIB();
	}

	return;
}

static LRESULT CALLBACK SubclassWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
	LRESULT lResult;

	switch (message) {
	case WM_SYSCOMMAND:
		//See if system menu options are enabled
		if (gbSystemMenuOptions == TRUE) {
			BOOL bRet;

			//Handle WinGlide system menu options
			bRet = HandleWinGlideSystemMenuOptions(hWnd, wParam, lParam);

			//Break if the message was handled
			if (bRet == TRUE) {
				break;
			}

			//If the message was not handled, pass it on to the previous window procedure
		}

		//Send the WM_SYSCOMMAND message to the previous window procedure
		return CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);

	case WM_MOVE:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);

		return lResult;

	case WM_SIZE:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);

		//Put the DIB on the screen
		DrawStretchedDIB();

		return lResult;

	case WM_DESTROY:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);

		//Unsubclass the window because it is being destroyed
		CWGUnsubclassWindowBecauseDestroyed();

		return lResult;

	default:
		return CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);
	}
	return 0;	
}

static BOOL CreateFrameUpdateThread(LPDWORD pdwFrameUpdateThreadTID, LPVOID pParameter) {
	//See if the thread has already been created
	if (hFrameUpdateThread != NULL) {
		DestroyFrameUpdateThread();
	}
	
	//Create the event objects needed for synchronization
	//Set to reset manually and initially be nonsignaled
	hMainThreadEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
	if (hMainThreadEvent == NULL) {
		//Free frame update thread objects
		DestroyFrameUpdateThread();
		return FALSE;
	}
	//Set to reset automatically and initially be nonsignaled
	hFrameUpdateThreadEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
	if (hFrameUpdateThreadEvent == NULL) {
		//Free frame update thread objects
		DestroyFrameUpdateThread();
		return FALSE;
	}
	//Set to reset automatically and initially be nonsignaled
	hFrameUpdateThreadExitedEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
	if (hFrameUpdateThreadExitedEvent == NULL) {
		//Free frame update thread objects
		DestroyFrameUpdateThread();
		return FALSE;
	}

	//Reset the flag that tells the frame update thread to exit
	gbTellFrameUpdateThreadToExit = FALSE;

	//Create the frame update thread
	hFrameUpdateThread = CreateThread(NULL, 0, FrameUpdateThreadProc, pParameter, 0, pdwFrameUpdateThreadTID);
	if (hFrameUpdateThread == NULL) {
		//Free frame update thread objects
		DestroyFrameUpdateThread();
		return FALSE;
	}

	return TRUE;
}

static VOID DestroyFrameUpdateThread(VOID) {
	BOOL bRet;

	//Tell the frame update thread to exit
	gbTellFrameUpdateThreadToExit = TRUE;

	if (hFrameUpdateThreadEvent != NULL) {
		//Tell the frame update thread to continue so that it will exit
		bRet = SetEvent(hFrameUpdateThreadEvent);
	}

	//Wait for the frame update thread to exit if it was created
	if (hFrameUpdateThread != NULL) {
		BOOL bRet;
		DWORD dwRet;

		//See if the thread has already exited
		bRet = GetExitCodeThread(hFrameUpdateThread, &dwRet);
		//If the thread is still active, wait for it to exit
		if ((bRet == TRUE) && (dwRet == STILL_ACTIVE)) {
			//Wait for the thread to exit
			dwRet = WaitForSingleObject(hFrameUpdateThreadExitedEvent, INFINITE);
		}
		//Close the thread handle
		CloseHandle(hFrameUpdateThread);
		hFrameUpdateThread = NULL;
	}

	//Free the event objects needed for synchronization
	if (hMainThreadEvent != NULL) {
		CloseHandle(hMainThreadEvent);
		hMainThreadEvent = NULL;
	}
	if (hFrameUpdateThreadEvent != NULL) {
		CloseHandle(hFrameUpdateThreadEvent);
		hFrameUpdateThreadEvent = NULL;
	}
	if (hFrameUpdateThreadExitedEvent != NULL) {
		CloseHandle(hFrameUpdateThreadExitedEvent);
		hFrameUpdateThreadExitedEvent = NULL;
	}

	return;
}

static DWORD WINAPI FrameUpdateThreadProc(LPVOID pParameter) {
	BOOL bRet;
	DWORD dwRet;

	while (1) {
		//Handle synchronization
		bRet = SetEvent(hMainThreadEvent);
		dwRet = WaitForSingleObject(hFrameUpdateThreadEvent, INFINITE);
		
		//See if the thread should exit
		if (gbTellFrameUpdateThreadToExit == TRUE) {
			break;
		}

		//Put the DIB on the screen
		DrawStretchedDIB();
	}
	
	//Set the event saying that the thread has exited
	SetEvent(hFrameUpdateThreadExitedEvent);

	//Exit the thread
	ExitThread(0);
	return 0;
}

